/*
* FileMaker Web Connector for Tebleau(Version 9.*, 10.0, 10.1)
* Author: FileMaker
* Note: This file is intended to be run in Tebleau Web Connector to import data from FileMaker solution
*/

(function() {

  var fmConnector = tableau.makeConnector();

  fmConnector.idRegExp = new RegExp('^[a-zA-Z0-9_]\*$');
  fmConnector.filterId = function(id) {
    return id.replace(/[^\w]+/g, '');
  }

  fmConnector.init = function(initCallback) {
    // If there is connectionData present in the interactive phase or Authentication phase, repopulate the input text box.
    // This is hit when re-login or editing a connection in Tableau.
    if (tableau.phase === tableau.phaseEnum.interactivePhase || tableau.phase === tableau.phaseEnum.authPhase ) {
      if(tableau.connectionData){
    		var conf  = JSON.parse(tableau.connectionData);
      		$('#solutionName').val(conf.solution);
      		$('#layoutName').val(conf.layout);
      		$('#user').val(tableau.username);
          $('#incremental').attr('checked', conf.incremental);
          $("#submitButton").prop('disabled', false);
      }
      $('#solutionName').focus();
    }
    //After Tableau close, tableau.password will be removed. Tableau is in data-gathering phase when it re-open and
    //it should call the connector in auth phase, which will display the UI that lets the user sign in again.
    if (tableau.phase === tableau.phaseEnum.authPhase) {
      //disable all input fields except username and password for re-login purpose.
        $(".inputBoxTitle h3").text(lang.Title_Login_Again)
        $('#solutionName').prop('disabled', true);
        $('#layoutName').prop('disabled', true);
        $('#incremental').prop('disabled', true);
        $('#user').focus();
    }

    // set tableau.alwaysShowAuthUI to true. This will make Tableau to display custom re-login UI when is re-open.
    tableau.alwaysShowAuthUI = true;

    initCallback();
  };

  fmConnector.getSchema = function(schemaCallback) {
    var conf  = JSON.parse(tableau.connectionData);
    schemaCallback([conf.tableInfo]);
  };

  fmConnector.createDataCursor = function(successCallback) {
    ///lastRecordToken is a string so it need to converted to number to match recordId data type.
  	var conf  = JSON.parse(tableau.connectionData);
    var passwordObj = fmConnector.getPasswordObj();
    // calling createCursor api
    var connectionUrl = conf.apiPath +"databases/" + encodeURIComponent(conf.solution) +"/layouts/" + encodeURIComponent(conf.layout) + "/cursor";
    //console.log("CREATE CUSRSOR TOKEN ");
    var xhr = $.ajax({
      url: connectionUrl,
      dataType: 'json',
      contentType: "application/json",
      headers: {"Authorization": "Bearer " + passwordObj.token},
      type:"POST",
      success: function (res, textStatus, xhr) {
        if (res.messages && res.messages[0].code === '0') {
            passwordObj.cursorToken = res.response.cursorToken;
            tableau.password = JSON.stringify(passwordObj);
            successCallback();
        } else {
          tableau.abortWithError(lang.Error_Create_Cursor_Failed+": " + xhr.responseText);
        }
      },
      error: function (xhr, textStatus, thrownError) {
        tableau.abortWithError(lang.Error_Create_Cursor_Failed+ " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
      }
    });
  };

  // Reset Cursor for incremantal extraction
  fmConnector.resetDataCursor = function(lastRecordId) {
	  	var conf  = JSON.parse(tableau.connectionData);
	    var passwordObj = fmConnector.getPasswordObj();
	    // calling createCursor api
	    var connectionUrl = conf.apiPath +"databases/" + encodeURIComponent(conf.solution) +"/layouts/" + encodeURIComponent(conf.layout) + "/cursor/reset";
	    //console.log("RESET CUSRSOR ");

	    var xhr = $.ajax({
	        url: connectionUrl,
	        dataType: 'json',
	        contentType: "application/json",
	        headers: {"Authorization": "Bearer " + passwordObj.token, "X-FM-Data-Cursor-Token":passwordObj.cursorToken },
	        type:"POST",
          data: lastRecordId===0 ? "" : JSON.stringify({recordId:lastRecordId.toString()}),
	        success: function (res, textStatus, xhr) {
	            if (res.messages && res.messages[0].code !== '0') {
	               tableau.abortWithError(lang.Error_Reset_Cursor +' :'+ xhr.responseText);
	            }
	        },
	        error: function (xhr, textStatus, thrownError) {
            if(xhr.readyState===4 && xhr.responseText.indexOf("952")>-1){ //handle Invalid token
              //Skip re-login which will be called in getTableData step.
            }
            else{
	            tableau.abortWithError(lang.Error_Reset_Cursor + " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
            }
	        },
	        async: false
	    });
	  };

  fmConnector.fetchRows = function(table, conf, passwordObj, lastRecordId, doneCallback, successCallback) {
    var pageSize = parseInt(conf.pageSize || 1000); //500 5000
    var connectionUrl = conf.apiPath + "databases/"+encodeURIComponent(conf.solution) + "/layouts/" + encodeURIComponent(conf.layout) + "/cursor?_limit="+pageSize;
    var xhr = $.ajax({
      url: connectionUrl,
      dataType: 'json',
      contentType: "application/json",
      headers: {"Authorization": "Bearer " + passwordObj.token, "X-FM-Data-Cursor-Token":passwordObj.cursorToken },
      success: function (res, textStatus, xhr)  {
        successCallback(res, xhr, conf, table, passwordObj, lastRecordId, pageSize, doneCallback);
      },
      error: function (xhr, textStatus, thrownError) {
        if(xhr.readyState===4 && xhr.responseText.indexOf("952")>-1){//handle Invalid token
          //If FM session expired during Tableau extracting data, we can relogin FM and pickup from lastRecordId.
          fmConnector.FMLogin(lastRecordId.toString(), table, doneCallback);
        }else{
          tableau.abortWithError(lang.Error_Failed_To_Fetch_Data + " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
        }
      }
    });
  };

  var fetchCallback = function(res, xhr, conf, table, passwordObj, lastRecordId, pageSize, doneCallback) {
    var toRet = [];
    if (res.messages && res.messages[0].code === '0') {
      if(res.response.data.length>0){
        res.response.data.forEach(function(record){
          var rec = util.dataToLocal(record.fieldData, conf.fieldTypes, conf.fieldNames);
          if (record.hasOwnProperty("recordId")) {
          	rec["recordId"] = parseInt(record.recordId);
          }
          toRet.push(rec);
          lastRecordId = record.recordId;
        });
        table.appendRows(toRet);
        if (toRet.length < pageSize) {
          doneCallback();
        } else {
          fmConnector.fetchRows(table, conf, passwordObj, toRet, doneCallback, fetchCallback);
        }
      } else {
        if(lastRecordId == 0){
          return tableau.abortWithError(lang.Error_No_Results_Found);
        } else {
          doneCallback();        
        }
        return false;
      }
    } else {
      tableau.abortWithError(lang.Error_Failed_To_Fetch_Data + " : " + xhr.responseText);
    }
  };

  //We make lastRecordToken looks like {lastRecordId:123, hasMore:false}
  fmConnector.getData = function(table, doneCallback) {
    var conf  = JSON.parse(tableau.connectionData);
    var passwordObj = fmConnector.getPasswordObj();
    //lastRecordToken is a string, either empty string or recordToken, so it need to converted to number to match recordId data type.
    var lastRecordId = parseInt(table.incrementValue || 0);
    fmConnector.resetDataCursor(lastRecordId);
    // to call fetch data api
    fmConnector.fetchRows(table, conf, passwordObj, lastRecordId, doneCallback, fetchCallback);
  };

  fmConnector.shutdown = function(shutdownCallback) {
    //In case of re-login cuased by expried token durign gatehrData phase,
    //new token can't be updated into tableau.password and won't be reusable when this connector was reloaded.
    //We have to enforce logout for each shutdown after re-login to avoid creating idel FM session.
    if(tableau.phase === tableau.phaseEnum.gatherDataPhase && fmConnector.reLogin){
        var passwordObj = fmConnector.getPasswordObj();
        var connectionConf  = JSON.parse(tableau.connectionData);
        var connectionUrl = connectionConf.apiPath + "databases/"+encodeURIComponent(connectionConf.solution)+"/sessions/"+passwordObj.token;
        var xhr = $.ajax({
            url: connectionUrl,
            type:"DELETE",
            headers: {"Authorization": "Bearer " + passwordObj.token},
            success: function (res, textStatus, xhr) {
              shutdownCallback();
            },
            error: function (xhr, textStatus, thrownError) {
                tableau.abortWithError(lang.Error_Logout_Failed + " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
                shutdownCallback();
            }
        })
    }else{
      shutdownCallback();
    }

  }

  /* helper functions */
  fmConnector.getPasswordObj = function(){
    if(!tableau.password){
      return tableau.abortWithError(lang.Error_Missing_Password_Object );
    }
    try {
      var passwordObj = JSON.parse(tableau.password);
      if(tableau.phase === tableau.phaseEnum.gatherDataPhase){
        if(!passwordObj.token){
          return tableau.abortWithError(lang.Error_Missing_Session_Token  );
        }
        if(!passwordObj.cursorToken){
          return tableau.abortWithError(lang.Error_Missing_Cursor_Token );
        }
      }
    } catch (e) {
      return tableau.abortWithError(lang.Error_Missing_Password_Object_Republish);
    }
    return passwordObj;
  }

  //store field names, types and other resource metaData into tableau.connectionData
  fmConnector.storeMetaData = function(metaData){
    var dataTypesMap = {"text":"string", "bool":"bool", "date":"date", "time":"string", "timeStamp":"datetime", "number":"float", "int":"int"};
    var connectionConf  = JSON.parse(tableau.connectionData);
    var cols = [];
    connectionConf.fieldNames = [];
    connectionConf.fieldTypes = [];

    metaData.forEach(function(meta){
      if(connectionConf.fieldNames.indexOf(meta.id) == -1 && dataTypesMap.hasOwnProperty(meta.result)){ //skip duplicated field if it is already included and check we support result type
        connectionConf.fieldNames.push(meta.id);
        connectionConf.fieldTypes.push(dataTypesMap[meta.result]);
        cols.push({
          id: meta.id,
          alias: meta.name,
          dataType: dataTypesMap[meta.result]
        });
      }
    });

    var tableInfo = {
      id: connectionConf.layout,
      columns: cols
    };
    // add recordId
    tableInfo.columns.push({
  	  id: "recordId",
	  alias: "recordId",
	  dataType: "int"
    });
    // Layout name doesn't conform to Tableau table id requirements
    // filter layout name for id and store original name in alias
    if (!fmConnector.idRegExp.test(tableInfo.id)) {
      var tableId = fmConnector.filterId(tableInfo.id);
      if (tableId.length == 0) {
        tableInfo.id = "table";
      } else {
        tableInfo.id = tableId;
      }
      tableInfo.alias = connectionConf.layout;
    }

    if(connectionConf.incremental && connectionConf.fieldNames.length>0){
      tableInfo.incrementColumnId = "recordId";
    }

    connectionConf.tableInfo = tableInfo;

    // save metadata into the configuration for the connection
    tableau.connectionData = JSON.stringify(connectionConf);
    // name the data source. This will be the data source name in Tableau
    tableau.connectionName = 'FM: ' + connectionConf.solution + '/ ' + connectionConf.layout;
  };

  fmConnector.getMetaData = function(successCallback) {
  	var connectionConf  = JSON.parse(tableau.connectionData);
    var connectionUrl = connectionConf.apiPath + "databases/"+encodeURIComponent(connectionConf.solution) +"/layouts/"+encodeURIComponent(connectionConf.layout)+"/metadata";
    var passwordObj = fmConnector.getPasswordObj();
  	var xhr = $.ajax({
      url: connectionUrl,
      type:"GET",
      headers: {"Authorization": "Bearer " + passwordObj.token},
      success: function (res, textStatus, xhr) {
        if (res.messages && res.messages[0].code === '0') {
          if(res.response.metaData.length==0){
            tableau.abortWithError(lang.Error_Get_Meta_Data_Failed +": "+ xhr.responseText);
          }else{
            fmConnector.storeMetaData(res.response.metaData);
            successCallback();
          }
        } else {
          tableau.abortWithError(lang.Error_Get_Meta_Data_Failed +": "+ xhr.responseText);
        }
        },
      error: function (xhr, textStatus, thrownError) {
        tableau.abortWithError(lang.Error_Get_Meta_Data_Failed + " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
      }
    });
  }

  //The optional string parameter lastRecordToken indicates that the wip session expired during Tableau extracting data.
  fmConnector.FMLogin = function(lastRecordToken, table, doneCallback) {
    var passwordObj = {};
    var connectionConf  = JSON.parse(tableau.connectionData);
    var connectionUrl = connectionConf.apiPath + "databases/"+encodeURIComponent(connectionConf.solution)+"/sessions";
    if(connectionConf.loginType === "oauth"){
      var headers = {
        "Authorization": "Fmid " + connectionConf.idToken
      };
    }else{//Regular login with FM account
      return tableau.abortWithError(lang.Error_Login_Failed + " : invalid login type");
    }
    var request = $.ajax({
      url: 'https://' + connectionConf.fmidHost + '/portal/mgmt-account/authentication/refreshtoken',
      dataType: 'json',
      contentType: "application/json",
      type:"POST",
      data: JSON.stringify({"refreshToken" : connectionConf.refreshToken}),
      success: function (res, textStatus, request) {
          if (res.errcode && res.errcode === 'Ok') {
			var xhr = $.ajax({
				url: connectionUrl,
				dataType: 'json',
				contentType: "application/json",
				headers: headers,
				type:"POST",
				data: {},
				success: function (res, textStatus, xhr) {
					if (res.messages && res.messages[0].code === '0') {
					  try{
						passwordObj.token = xhr.getResponseHeader('x-fm-data-access-token');
						tableau.password = JSON.stringify(passwordObj);
						fmConnector.createDataCursor(function() {
						  if(tableau.phase === tableau.phaseEnum.gatherDataPhase){
							fmConnector.reLogin = true;
							// Reload passwordObj since we got new token and cursor token
							passwordObj = fmConnector.getPasswordObj();
							//Re-login during a Tableau session, skip setup metadata and getTableData directly
							fmConnector.fetchRows(table, connectionConf, passwordObj, lastRecordToken, doneCallback, fetchCallback);
						  } else {
							fmConnector.getMetaData(function() {
							  $('#loader').hide();
							  tableau.submit();
							});
						  }
						});
					  }catch(err){
							$('#loader').hide();
							return tableau.abortWithError(err.message);
					  }
					} else {
						return tableau.abortWithError(lang.Error_Login_Failed +": " + xhr.responseText);
					}
				},
				error: function (xhr, textStatus, thrownError) {
				  $('#loader').hide();
				  return  tableau.abortWithError(lang.Error_Login_Failed + " : " +util.makeErrorMessage(xhr, textStatus, thrownError));
				}
          });
                } else {
              return tableau.abortWithError(lang.Error_Login_Failed +" : " + request.responseText);
          }
      },
      error: function (request, textStatus, thrownError) {
        $('#loader').hide();
        return  tableau.abortWithError(lang.Error_Login_Failed + " : " +util.makeErrorMessage(request, textStatus, thrownError));
      }
    });
  }

  fmConnector.Oauth = {

      providers : [],  //contains oauth providers meta data which will be populated with getProvidersInfo()

      getProvidersInfo : function(){
        // var xhr = $.ajax({
        //     context: this,
        //     dataType:'json',
        //     url: location.origin+'/fmws/oauthproviderinfo',
        //     success: function (res, textStatus, xhr)  {
        //       if(res.data){
        //         this.providers = res.data.Provider;
        //         this.providers.sort(function SortByName(a, b){//sort on provider name to make them rendered in consistent order.
        //           var aName = a.Name.toLowerCase();
        //           var bName = b.Name.toLowerCase();
        //           return ((aName < bName) ? -1 : ((aName > bName) ? 1 : 0));
        //         });
        //         this.renderBtns();
        //       }
        //     },
        //     error: function (xhr, textStatus, thrownError) {
        //       tableau.abortWithError(lang.Error_OAuth_Fail_At_GetProvidersInfo+ util.makeErrorMessage(xhr, textStatus, thrownError));
        //     }
        //   })
        this.providers = [
          {
            "Name":"FileMaker ID"
          }
        ];
        this.renderBtns();
      },

      renderBtns : function(){
        var providersList = $('#oauth-container');

        if(this.providers.length>0){
          this.providers.forEach(function(provider){
            var $btn = $("<button>", {"data-provider-name":provider.Name, "class": "oauth-btn", "text": provider.Name});
                $btn.click(fmConnector.Oauth.doOauth);
            providersList.append($btn);
          })

          util.getCookie("oAuthIdentifier") ? providersList.show() : providersList.slideDown();
        }
      },

      doOauth: function(e){
        var connectionConf = {};
        connectionConf.loginType = "oauth";
        connectionConf.apiPath = location.origin;
        connectionConf.apiPath = connectionConf.apiPath + '/fmi/data/vLatest/';
        connectionConf.solution = $('#solutionName').val().trim();
        connectionConf.layout = $('#layoutName').val().trim();
        connectionConf.incremental = $('#incremental').is(':checked');
        tableau.connectionData = JSON.stringify(connectionConf);
        localStorage.setItem('connData', tableau.connectionData);
        tableau.username = "";

        if(util.validateInput()===true){
          fmConnector.Oauth.getOauthUrl($(this).data("provider-name"));
        }
        return false;
      },

      getOauthUrl: function(provider){
        $('#loader').show();
        var callbackUrl = location.href;
        var apiUrl = location.origin + '/fmi/data/vLatest/fmidurl';
        var xhr = $.ajax({
            context: this,
            url: apiUrl,
            dataType:"text",
            success: function (data, textStatus, xhr)  {
              $('#loader').hide();
              location.href = JSON.parse(xhr.responseText).url;
            },
            error: function (xhr, textStatus, thrownError) {
              $('#loader').hide();
              tableau.abortWithError(lang.Error_Login_Failed + " : " + util.makeErrorMessage(xhr, textStatus, thrownError))
            }
          })
      }
  };

  tableau.registerConnector(fmConnector);

  //
  // Setup connector UI
  //

  $(window).load(function() {
    if(tableau.phase === tableau.phaseEnum.gatherDataPhase ){
      return;
    }

    util.applyI18nString();

    var query = util.queryString();
    if(query.code){
      tableau.connectionData = localStorage.getItem('connData');
      if(!tableau.connectionData) { // If user clicking on link of FMS oauth redirect in broswer history, force it back to connector page.
        return location.replace(location.origin+location.pathname);
      }
      fmConnector.Oauth.getProvidersInfo();
      var apiUrl = location.origin + '/fmi/data/vLatest/tokenexchange';
      var xhr = $.ajax({
          context: this,
          url: apiUrl,
          method: "POST",
          data: 'code=' + encodeURIComponent(query.code),
          success: function (data, textStatus, xhr)  {
            var connectionConf  = JSON.parse(tableau.connectionData);
            var resp = JSON.parse(xhr.responseText);
            connectionConf.idToken = resp.token;
            connectionConf.refreshToken = resp.refresh;
            tableau.connectionData = JSON.stringify(connectionConf);
            var endpointReq = $.ajax({
              url: location.origin + '/fmi/data/vLatest/apihost',
              dataType: 'json',
              contentType: "application/json",
              type:"GET",
              success: function (res, textStatus, endpointReq) {
                var endpointInfo = res;
                var connectionConf  = JSON.parse(tableau.connectionData);
                connectionConf.fmidHost = endpointInfo.API_Host;
                tableau.connectionData = JSON.stringify(connectionConf);
                fmConnector.FMLogin();
              },
              error: function (endpointReq, textStatus, thrownError) {
                $('#loader').hide();
                return  tableau.abortWithError(lang.Error_Login_Failed + " : " +util.makeErrorMessage(endpointReq, textStatus, thrownError));
              }
            });
          },
          error: function (xhr, textStatus, thrownError) {
            // $('#loader').hide();
            tableau.abortWithError(lang.Error_Login_Failed + util.makeErrorMessage(xhr, textStatus, thrownError))
          }
        });
      // var d = new Date();
        // d.setTime(d.getTime() + (30*1000));
        // document.cookie = "oAuthIdentifier="+query.identifier+"; expires="+d.toUTCString();
        // return location.replace(location.origin+location.pathname);
    }else if(util.getCookie("oAuthIdentifier") && util.getCookie("oAuthRequestId")){  // this is redirected after putting identifier in cookie
              fmConnector.Oauth.getProvidersInfo();
              tableau.password = JSON.stringify({oAuthRequestId: util.getCookie("oAuthRequestId"), oAuthIdentifier: util.getCookie("oAuthIdentifier"), token:''});
              util.removeCookie("oAuthIdentifier");
              util.removeCookie("oAuthRequestId");
              var connectionConf  = JSON.parse(tableau.connectionData);
              $('#solutionName').val(connectionConf.solution);
              $('#layoutName').val(connectionConf.layout);
              $('#incremental').attr('checked', connectionConf.incremental);
              $("#submitButton").prop('disabled', false);
              $('#loader').show();
              fmConnector.FMLogin();
    }else {
      fmConnector.Oauth.getProvidersInfo();
    }

    if(!tableau.connectionData){
      //Initial page load should not have saved connection data and submitButton should be disabled.
      //If page are reloaded after login failure, submitButton should be enabled by init().
      $("#submitButton").prop('disabled', true);
    }
    $('#solutionName').on('input', util.enableBtnOnInput);
    $('#layoutName').on('input', util.enableBtnOnInput);
    $("#submitButton").click(function() {
      var connectionConf = {};
      connectionConf.loginType = "";
      connectionConf.apiPath = location.origin;
      connectionConf.apiPath = connectionConf.apiPath + '/fmi/data/vLatest/';
      connectionConf.solution = $('#solutionName').val().trim();
      connectionConf.layout = $('#layoutName').val().trim();
      connectionConf.incremental = $('#incremental').is(':checked');
      tableau.connectionData = JSON.stringify(connectionConf);
      // tableau.username = $('#user').val().trim();
      tableau.username = '';
      // tableau.password = JSON.stringify({password: $('#password').val().trim(), token:''});
      tableau.password = JSON.stringify({password: '', token: '', idToken: $('#token').val().trim()});

      if(tableau.phase === tableau.phaseEnum.interactivePhase || tableau.phase === tableau.phaseEnum.authPhase ){
        if(util.validateInput()){
          $('#loader').toggle(0);
          fmConnector.FMLogin();
        }
      }
    });
    //$("body").append(navigator.userAgent);
  });

})();
